/* Copyright (C) 2002-2018 RealVNC Ltd. All Rights Reserved.
*/

#ifndef UUID_324cd44a_4435_4288_b4fb_45b83430e2e7
#define UUID_324cd44a_4435_4288_b4fb_45b83430e2e7

/**
 * \file vncdapsdk.h
 * 
 * This is the main header file for the VNC DAP SDK. Applications using the
 * SDK should include only this file.
 *
 * This file includes \ref vncint.h. Applications including vncdapsdk.h must
 * therefore define either VNC_USE_DEFAULT_INT_TYPEDEFS or VNC_USE_STDINT_H,
 * as described in vncint.h.
 */

/**
 * \mainpage VNC DAP SDK
 *
 * \section section_contents Contents
 *
 *  -# \ref section_overview
 *  -# \ref section_usage
 *  -# \ref section_references
 *  -# \ref section_legal
 *
 * \section section_overview Overview
 *
 * VNC DAP SDK implements the client side of MirrorLink's Device Attestation
 * Protocol, as defined in \ref ml_dap "[ML-DAP]".
 *
 * The application must be able to provide VNC DAP SDK with the following
 * information:
 *
 *  - the MirrorLink version implemented by the MirrorLink server
 *  - the URL of the DAP server
 *  - the trust root certificate that will be used to validate the DAP server's
 *    certificate
 *
 * The MirrorLink server version can be found in the root device XML of the
 * MirrorLink server. The DAP server URL can be obtained from the
 * TmApplicationServerService, in its response to a LaunchApplication action
 * (see \ref ml_app "[ML-APP]"). MirrorLink UPnP is implemented by VNC
 * Discovery SDK.
 *
 * \section section_usage Basic usage
 *
 * \subsection subsection_usage_compiling Compiling your application
 *
 * To use the SDK, your application must first define exactly one of the
 * preprocessor macros VNC_USE_STDINT_H or VNC_USE_DEFAULT_INT_TYPEDEFS.  These
 * macros govern the definitions of the vnc_*_t integer types defined by
 * vncint.h.
 *
 *  - If your build environment provides the ISO C99 header stdint.h, or you
 *    can author an equivalent yourself, you should define VNC_USE_STDINT_H.
 *    This is the recommended approach for defining the SDK's integer types.
 *  - If you examine vncint.h and find that the types used when you define
 *    VNC_USE_DEFAULT_INT_TYPEDEFS are correct for your platform, then you may
 *    wish to define VNC_USE_DEFAULT_INT_TYPEDEFS instead of VNC_USE_STDINT_H.
 *    Additionally, if you are targeting a 64-bit platform with a compiler
 *    other than Microsoft Visual C++ or gcc, then you may have to define
 *    VNC_FORCE_64_BIT in order to define vnc_intptr_t and vnc_uintptr_t
 *    correctly.  See vncint.h itself for further information.
 *
 * The types defined by VNC_USE_DEFAULT_INT_TYPEDEFS are known to be correct
 * for Microsoft Visual C++ 2005.
 *
 * After defining either VNC_USE_STDINT_H or VNC_USE_DEFAULT_INT_TYPEDEFS, your
 * application should include vncdapsdk.h.  You are encouraged to use the
 * VNCVerifyIntegerTypes() macro to verify that the integer types have been
 * defined correctly.
 *
 * \subsection subsection_usage_startup Initializing the SDK
 *
 * -# Load the SDK DLL (necessary only if you did not link against it).
 *    - on Windows, call LoadLibrary()
 *    - on Linux, call dlopen()
 * -# Locate the SDK entry point, VNCDAPSDKInitialize().
 *    - on Windows, call GetProcAddress()
 *    - on Linux, call dlsym()
 * -# Call VNCDAPSDKInitialize().  This populates a VNCDAPSDK structure with
 *   pointers to the rest of the SDK's 'methods'.
 *
 * Typically, your application will call VNCDAPSDKInitialize() at start-up.
 * You should not call VNCDAPSDKInitialize() more than once.
 *
 * \subsection subsection_usage_creating Creating a DAP client object
 *
 * Call VNCDAPClientCreate() to create a DAP client object. You may reuse the
 * same DAP client object for a succession of DAP sessions that require a
 * similar configuration. There is no need to create a new DAP client for each
 * session.
 *
 * When you create a DAP client object, you supply pointers to various
 * callbacks that provide your application with information about DAP events.
 * It is recommended that you implement VNCDAPClientLogCallback(), so that you
 * are given detailed information about any attestation failures.
 *
 * After creating a DAP client object, you may find it useful to use
 * VNCDAPClientSetContext() to associate some application-specific data with
 * the DAP client. For example, you could pass a pointer to a struct that holds
 * state for your application (window handles, log file handle, etc). It is
 * recommended that any call to VNCDAPClientSetContext() is made immediately
 * after VNCDAPClientCreate(), so that the context pointer is available to all
 * callbacks that the SDK may make.
 *
 * \subsection subsection_usage_licensing Licensing the DAP client
 *
 * In order to use the SDK, the application must always provide each DAP client
 * instance with at least one license. Use VNCDAPClientAddLicense() to do this.
 * Contact your RealVNC sales representative to obtain a license.
 *
 * \subsection subsection_usage_configuring Configuring the DAP client
 *
 * Before a DAP client can perform an attestation request, it must be
 * configured. The application must provide each DAP client with the following
 * information:
 *
 *  - The DAP server URL. This is the URL returned by the UPnP Application
 *    Server Service in the response to a successful LaunchApplication action
 *    (see \ref ml_app "[ML-APP]"). See VNCDAPClientSetServerURL() for further
 *    information.
 *  - The DAP trust root certificate. This is the certificate that will be used
 *    to verify the chain of device and manufacturer certificates received in
 *    each &lt;attestationResponse&gt; (see \ref ml_dap "[ML-DAP]"). See
 *    VNCDAPClientSetTrustRoot() for further information.
 *
 * \subsection subsection_usage_request Making an attesation request
 *
 * The application makes attestation requests by calling
 * VNCDAPClientAttestationRequest(). This method blocks the calling thread
 * until either the DAP server has responded or the configured timeout elapses.
 *
 * The application should check the return value of
 * VNCDAPClientAttestationRequest() to determine whether the attestation was
 * successful. If the attestation is successful, then a pointer to a
 * ::VNCDAPAttestationResponse structure is returned. The application is
 * responsible for freeing this structure by calling
 * VNCDAPAttestationResponseDestroy() when it is no longer required.
 *
 * The application may re-use the same DAP client object for multiple calls to
 * VNCDAPClientAttestationRequest().
 *
 * \subsection subsection_usage_destroying Destroying a DAP client
 *
 * Use VNCDAPClientDestroy() to destroy a DAP client when it is no longer
 * required.
 *
 * \subsection subsection_usage_threading Threading
 *
 * VNCDAPClient instances do not create background threads. All VNC DAP SDK
 * calls are synchronous and block the calling thread until they return.
 *
 * VNCDAPClientAttestationRequest() issues a request to the DAP server and then
 * waits for either a reponse or a timeout. For this reason, the application
 * should not call VNCDAPClientAttestationRequest() from a thread that may not
 * safely be blocked.
 *
 * All calls to VNCDAPClient callbacks are made synchronously from the
 * application thread that calls the API that generates the callback.
 *
 * Once created, VNCDAPClient instances are safe for use by any thread, but
 * only by one thread at a time. The application must not perform operations on
 * the same VNCDAPClient instance from more than one thread at the same time.
 *
 * \section section_references References
 * 
 *  -# \anchor ml_dap [ML-DAP] CCC-TS-014 Device Attestation Protocol,
 *     versions 1.1 and 1.2
 *  -# \anchor ml_app [ML-APP] CCC-TS-024 UPnP Application Server Service,
 *     versions 1.1 and 1.2
 *
 * \section section_legal Legal information
 *
 * Copyright &copy; 2002-2018 RealVNC Ltd.  All Rights Reserved.
 *
 * Details of and copyright notices for third-party software that is used by
 * the VNC DAP SDK can be found in the file Acknowledgements.txt in the SDK
 * distribution.
 *
 * RealVNC and VNC are trademarks of RealVNC Limited and are protected by
 * trademark registrations and/or pending trademark applications in the
 * European Union, United States of America and other jurisdictions.
 *
 * MirrorLink is a registered trademark of Car Connectivity Consortium in the
 * U.S. and in other countries.
 *
 * Other trademarks are the property of their respective owners.
 *
 * Protected by UK patents 2481870, 2491657; US patents 8760366, 9137657; EU patent 2652951.
 *
 * \see VNCVerifyIntegerTypes, VNCDAPSDK, VNCDAPSDKInitialize,
 * VNCDAPSDKUninitialize, VNCDAPClientCreate
 */

#include "vnccall.h"
#include "vncint.h"

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \brief A DAP client.
 *
 * This type is opaque. Use the VNC DAP SDK APIs to interact with it.
 *
 * Use VNCDAPClientCreate() to create a DAP client. 
 *
 * VNCDAPClient instances are safe for use by any thread, but only by one
 * thread at a time. The application must not perform operations on the same
 * VNCDAPClient instance from more than one thread at the same time.
 */
typedef struct VNCDAPClientImpl VNCDAPClient;

/**
 * \brief Error codes that may be returned by the DAP SDK API.
 *
 * The constants whose names begin with VNCDAPResult originate in the
 * &lt;result&gt; tag of an &lt;attestationResponse&gt; returned by a DAP
 * server, and may only be returned by VNCDAPClientAttestationRequest().
 *
 * The constants whose names begin with VNCDAPError are the result of a local
 * failure. They do not originate with the DAP server, but may arise when
 * processing a response that has been received from the DAP server (e.g. if
 * certificate validation fails).
 *
 * In all cases, there may be a more detailed description of the failure in
 * the log.
 *
 * \see VNCDAPGetErrorName(), VNCDAPClientLogCallback(),
 * VNCDAPClientAttestationRequest()
 */
typedef enum
{
    /**
     * \brief The operation was successful.
     */
    VNCDAPErrorNone = 0,

    /**
     * \brief The MirrorLink server does not implement the component that it
     * was asked to attest in the call to VNCDAPClientAttestationRequest().
     *
     * Returned by the DAP server.
     */
    VNCDAPResultComponentNotImplemented = 1,

    /**
     * \brief The DAP server does not implement the version of the DAP
     * protocol that was specified when VNCDAPClientCreate() was called.
     *
     * The MirrorLink client application can avoid this error by specifying
     * the version number that was returned by the MirrorLink server in its 
     * TmServerDevice XML.
     *
     * Returned by the DAP server.
     */
    VNCDAPResultVersionUnsupported = 2,

    /**
     * \brief The DAP server does not recognize the trust root certificate
     * that was provided in the last call to VNCDAPClientSetTrustRoot().
     *
     * Returned by the DAP server.
     */
    VNCDAPResultUnknownTrustRoot = 3,

    /**
     * \brief The DAP server does not recognize the component ID that was
     * specified in the call to VNCDAPClientAttestationRequest().
     *
     * Returned by the DAP server.
     */
    VNCDAPResultUnknownComponent = 4,

    /**
     * \brief Attestation failed, due to some server-side error.
     * 
     * Returned by the DAP server.
     */
    VNCDAPResultAttestationFailed = 5,

    /**
     * \brief Lowest error value that can be returned by the DAP SDK but that
     * does not originate with a DAP server.
     */
    VNCDAPErrorLOCAL = 1000,

    /**
     * \brief The operation failed for an unspecified reason.
     * 
     * Whereever possible, the DAP SDK will attempt to provide a more specific
     * error code.
     */
    VNCDAPErrorFailed = VNCDAPErrorLOCAL,

    /**
     * \brief The application called a DAP SDK API with an invalid parameter.
     */
    VNCDAPErrorInvalidParameter = VNCDAPErrorLOCAL + 1,

    /**
     * \brief A dynamic memory allocation failed.
     */
    VNCDAPErrorOutOfMemory = VNCDAPErrorLOCAL + 2,

    /**
     * \brief The DAP protocol version requested by a call to
     * VNCDAPClientCreate() is not supported by this release of the DAP SDK.
     */
    VNCDAPErrorServerVersionNotSupported = VNCDAPErrorLOCAL + 3,

    /**
     * \brief The DAP SDK license provided in the call to
     * VNCDAPClientAddLicense() is not valid.
     */
    VNCDAPErrorLicenseNotValid = VNCDAPErrorLOCAL + 4,

    /**
     * \brief The DAP SDK is not licensed to perform the requested operation.
     *
     * This could be because:
     *
     *  - The application has not called VNCDAPClientAddLicense()
     *  - The license added with VNCDAPClientAddLicense() has expired
     *  - The license added with VNCDAPClientAddLicense() does not grant
     *    license to perform the requested operation
     */
    VNCDAPErrorFeatureNotLicensed = VNCDAPErrorLOCAL + 5,

    /**
     * \brief The application called VNCDAPClientAttestationRequest() before it
     * made a successful call to VNCDAPClientSetServerURL().
     */
    VNCDAPErrorNoServerURL = VNCDAPErrorLOCAL + 6,

    /**
     * \brief The application called VNCDAPClientAttestationRequest() before it
     * made a successful call to VNCDAPClientSetTrustRoot().
     */
    VNCDAPErrorNoTrustRoot = VNCDAPErrorLOCAL + 7,

    /**
     * \brief An OS error occurred while communicating with the DAP server.
     */
    VNCDAPErrorNetworkError = VNCDAPErrorLOCAL + 8,

    /**
     * \brief The DAP server returned an &lt;attestationResponse&gt; that could
     * not be parsed.
     *
     * This could be because:
     *
     *  - The XML is not well-formed
     *  - The XML is not a valid &lt;attestationResponse&gt;
     *  - A CDATA element that should contain Base-64-encoded data (e.g.
     *    &lt;deviceCertificate&gt;) does not contain valid Base-64-encoded
     *    data
     */
    VNCDAPErrorInvalidAttestationResponse = VNCDAPErrorLOCAL + 9,

    /**
     * \brief An X.509 certificate is not valid.
     *
     * When returned by VNCDAPClientSetTrustRoot(), this error indicates that
     * the data provided by the application is not a valid X.509 certificate.
     * 
     * When returned by VNCDAPClientAttestationRequest(), this error indicates
     * that the DAP server returned a device certificate or a manufacturer
     * certificate that is not a valid X.509 certificate.
     */
    VNCDAPErrorInvalidCertificate = VNCDAPErrorLOCAL + 10,

    /**
     * \brief The DAP server returned an &lt;attestationResponse&gt; that was
     * syntactically valid, and which contained valid X.509 certificates, but
     * which failed one or more validation checks.
     *
     * This could be because:
     *
     *  - The device certificate does not have the required
     *    tcp-kp-AIKCertificate Extended Key Usage X.509 v3 extension
     *  - A manufacturer certificate does not have the CA:TRUE Basic
     *    Constraints X.509 v3 extension
     *  - Either the device certificate or a manufacturer certificate has
     *    expired
     *  - The device certificate and the manufacturer certificates do not form
     *    a chain of trust that ends in the trust root provided by the last
     *    successful call to VNCDAPClientSetTrustRoot()
     *  - A &lt;quoteSignature&gt; contained in an &lt;attestation&gt; is not a
     *    valid SHA1-with-RSA signature over the corresponding
     *    &lt;quoteInfo&gt;
     *  - The application specified a particular component ID (i.e. not
     *    ::VNCDAPComponentAll) in its call to
     *    VNCDAPClientAttestationRequest(), but that component ID is not
     *    attested in the response from the DAP server
     *
     * Check the log for a detailed desription of the failure.
     */
    VNCDAPErrorServerNotTrusted = VNCDAPErrorLOCAL + 11,

    /**
     * \brief The configured timeout elapsed before a complete
     * &lt;attestationResponse&gt; was received from the server.
     * 
     * To configure the timeout for attestation requests, use
     * VNCDAPClientSetTimeout().
     */
    VNCDAPErrorTimedOut = VNCDAPErrorLOCAL + 12,

    /**
     * \brief A non-blocking request is already pending.
     *
     * The DAP client currently only supports a single outstanding request
     * and making any further requests, either blocking or non-blocking,
     * whilst this is still pending will result in this error.
     * Similarly, attributes such as server URL and timeout cannot
     * be changed when a request is pending.
     */
    VNCDAPErrorRequestPending = VNCDAPErrorLOCAL + 13,

    /**
     * \brief Operation was aborted at user request.
     *
     * This error is returned if the user requests a disconnect
     * while a non-blocking operation is in progress.
     */
    VNCDAPErrorRequestAborted = VNCDAPErrorLOCAL + 14,

} VNCDAPError;

/**
 * \brief Statuses that may be reported to the application.
 *
 * These status codes indicate changes in the status of the DAP client
 * which are notified to the application via the VNCDAPClientStatusCallback().
 */
typedef enum
{
  VNCDAPClientStatusConnected = 1,
  VNCDAPClientStatusDisconnected = 2
} VNCDAPClientStatus;

/**
 * \defgroup VNCDAPComponent VNCDAPComponent constants
 *
 * Identifiers of components that may be attested.
 *
 * These constants are for use in calls to VNCDAPClientAttestationRequest().
 *
 * @{
 */

/** \brief The VNC Server. */
#define VNCDAPComponentTerminalModeVNCServer "TerminalMode:VNC-Server"

/** \brief The UPnP server. */
#define VNCDAPComponentTerminalModeUPnPServer "TerminalMode:UPnP-Server"

/** \brief The RTP Server. */
#define VNCDAPComponentTerminalModeRTPServer "TerminalMode:RTP-Server"

/** \brief The RTP client. */
#define VNCDAPComponentTerminalModeRTPClient "TerminalMode:RTP-Client"

/** \brief The CDB endpoint. */
#define VNCDAPComponentMirrorLinkCDBEndpoint "MirrorLink:CDB-Endpoint"

/** \brief The MirrorLink device itself. */
#define VNCDAPComponentMirrorLinkDevice "MirrorLink:Device"

/** \brief The WFD RTSP source. */
#define VNCDAPComponentMirrorLinkWFDRTSP "MirrorLink:WFD:RTSP"

/** \brief The HSML source. */
#define VNCDAPComponentMirrorLinkHSML "MirrorLink:HSML"

/** \brief Wildcard matching all DAP component IDs. */
#define VNCDAPComponentAll "*"

/** @} */

/** \brief An array of bytes. */
typedef struct
{
    /** \brief Pointer to the start of the array. */
    const vnc_uint8_t *data;

    /** \brief Length of the array. */
    size_t length;
} VNCDAPByteArray;

/**
 * \brief A single &lt;attestation&gt; in a parsed &lt;attestationResponse&gt;.
 */
typedef struct
{
    /**
     * \brief The ID of the MirrorLink component that is attested by this
     * attesation.
     */
    const char *componentId;

    /** \brief The MirrorLink server's URL for this component. */
    const char *url;

    /** \brief The application public key for this component. */
    VNCDAPByteArray applicationPublicKey;

    /** \brief The application public key for this component, Base-64 encoded. */
    const char *applicationPublicKeyBase64;
} VNCDAPAttestation;

/** \brief A parsed &lt;attestationResponse&gt;. */
typedef struct
{
    /**
     * \brief Array of pointers to received &lt;attestation&gt;s.
     */
    const VNCDAPAttestation *attestations;

    /**
     * \brief Size of the attestations array.
     */
    size_t attestationCount;

    /**
     * \brief Received device certificate, as a DER-encoded byte array (the SDK
     * performs the Base-64 decoding).
     *
     * As part of VNCDAPClientAttestationRequest(), the SDK validates the
     * device certificate using the manufacturer certificates and the trust
     * root. It is not necessary for the application to repeat this step.
     */
    VNCDAPByteArray deviceCertificate;

    /**
     * \brief Array of pointers to received manufacturer certificates, each of
     * which is a DER-encoded byte array (the SDK performs the Base-64
     * decoding).
     */
    const VNCDAPByteArray *manufacturerCertificates;

    /**
     * \brief Size of the manufacturerCertificates array.
     */
    size_t manufacturerCertificateCount;
} VNCDAPAttestationResponse;

/** \brief Extra information relating to an attestation response. */
typedef struct
{
    /**
     * \brief The nonce that was generated for the attestation request.
     */
    VNCDAPByteArray nonce;

} VNCDAPAttestationResponseInfo;

/**
 * \brief Called by the SDK to allow logging from the SDK's internals.
 *
 * The information logged by this callback is intended for use by application
 * developers and RealVNC support engineers.  It is not localized.
 *
 * The severity parameter is an integer from 0 to 100 inclusive.  The SDK
 * defines these logging severities, but may use other values in between:
 *
 *  -   0 - errors
 *  -  10 - important messages
 *  -  30 - informational messages
 *  - 100 - verbose debugging
 *
 * \param pVNCDAPClient The DAP client that has generated the log output.
 * \param pContext The context pointer associated with the DAP client by
 * VNCDAPClientCreate() or VNCDAPClientSetContext().
 * \param category The category of the log message, as a NUL-terminated UTF-8
 * encoded string.
 * \param severity The severity of the log message.
 * \param text The text of the log message, as a NUL-terminated UTF-8 encoded
 * string.
 *
 * \see VNCDAPClientCallbacks, VNCDAPClientCreate(), VNCDAPClientSetContext()
 */
typedef void VNCCALL
VNCDAPClientLogCallback(VNCDAPClient *pVNCDAPClient,
                        void *pContext,
                        const char *category,
                        vnc_int32_t severity,
                        const char *text);

/**
 * \brief Called by the SDK on client status changes.
 *
 * Currently this will be called to notify the application of the client
 * starting, connecting, disconnecting and stopping.
 *
 * \param pVNCDAPClient The DAP client that has generated the status.
 * \param pContext The context pointer associated with the DAP client by
 * VNCDAPClientCreate() or VNCDAPClientSetContext().
 * \param status The event in question.
 */
typedef void VNCCALL
VNCDAPClientStatusCallback(VNCDAPClient *pVNCDAPClient,
                           void *pContext,
                           VNCDAPClientStatus status);

/**
 * \brief Called by the SDK when its thread has started.
 *
 * This callback is called by the SDK thread before performing any other
 * operation.
 */
typedef void VNCCALL
VNCDAPClientThreadStartedCallback(VNCDAPClient *pVNCDAPClient,
                                  void *pContext);

/**
 * \brief Called by the SDK when its thread is just about to stop.
 *
 * This callback is called as the final act of the SDK thread just before
 * it terminates.
 */
typedef void VNCCALL
VNCDAPClientThreadStoppedCallback(VNCDAPClient *pVNCDAPClient,
                                  void *pContext);

/**
 * \brief Called by the SDK to deliver a non-blocking attestion response.
 *
 * This function is invoked by the SDK to deliver the response to a
 * VNCDAPClientAttestationRequestEx() call. The parameters are equivalent
 * to the output parameters and return code of VNCDAPClientAttestationRequest().
 *
 * The pAttestationResponse and attestationResponseSize parameters may be NULL
 * and zero respectively if an error occurred.
 *
 * \param pVNCDAPClient The DAP client that has generated the response.
 * \param pContext The context pointer associated with the DAP client by
 * VNCDAPClientCreate() or VNCDAPClientSetContext().
 * \param componentId The component passed to VNCDAPClientAttestationRequestEx().
 * \param result Equivalent to the return code of VNCDAPClientAttestationRequest().
 * \param pAttestationResponse A pointer to the attestation response, or
 * NULL if attestation failed.
 * \param attestationResponseSize The size of the structure pointed to by
 * pAttestationResponse, or 0 if attestation failed.
 * \param pInfo A pointer to extra information relating to the attestation
 * response, or NULL if attestation failed.
 * \param infoSize the size of the attestatation request info structure, or
 * 0 if attestation failed.
 *
 * \see VNCDAPClientCallbacks, VNCDAPClientCreate(), VNCDAPClientSetContext()
 */
typedef void VNCCALL
VNCDAPClientAttestationResponseCallback(VNCDAPClient *pVNCDAPClient,
                                        void *pContext,
                                        const char *componentId,
                                        VNCDAPError result,
                                        VNCDAPAttestationResponse *pAttestationResponse,
                                        size_t attestationResponseSize,
                                        VNCDAPAttestationResponseInfo *pInfo,
                                        size_t infoSize);

/**
 * \brief Structure that holds pointers to callbacks invoked by the SDK.
 *
 * Any of the pointers in this structure may be NULL if you choose not to
 * implement a particular callback.
 *
 * For a description of each of the callbacks, please see the documentation for
 * the corresponding typedef.
 *
 * \see VNCDAPClientCreate()
 */
typedef struct
{
  /** Called by the SDK to allow logging from the SDK's internals. */
  VNCDAPClientLogCallback *pLogCallback;
  /** Called by the SDK to notify the application of status changes. */
  VNCDAPClientStatusCallback *pStatusCallback;
  /** Called by the SDK to indicate its thread has started. */
  VNCDAPClientThreadStartedCallback *pThreadStartedCallback;
  /** Called by the SDK to indicate its thread has stopped. */
  VNCDAPClientThreadStoppedCallback *pThreadStoppedCallback;
  /** Called by the SDK to return attestation response. */
  VNCDAPClientAttestationResponseCallback *pAttestationResponseCallback;
} VNCDAPClientCallbacks;

/**
 * \brief Uninitializes the SDK.
 *
 * The application should call this function after destroying all DAP clients
 * and attestation responses but before unloading the SDK.
 *
 * \see VNCDAPSDKInitialize(), VNCDAPClientDestroy(),
 * VNCDAPAttestationResponseDestroy()
 */
typedef void VNCCALL
VNCDAPSDKUninitialize(void);

/**
 * \brief Returns a symbolic name corresponding to a DAP SDK error code.
 *
 * For example, if error is VNCDAPErrorInvalidParameter, then the returned
 * string is "InvalidParameter". Such strings may be useful as
 * internationalization keys or in log entries.
 *
 * The strings returned by VNCDAPGetErrorName() are brief and are not
 * internationalized. They are therefore not suitable for display to an end
 * user.
 * 
 * \param error The error code.
 * \return A short UTF-8 string corresponding to the error code. The memory
 * pointed to is owned by the SDK; do not free it. If the error code is not
 * recognized by the SDK, then the return value is NULL.
 */
typedef const char *VNCCALL
VNCDAPGetErrorName(VNCDAPError error);

/**
 * \brief Creates a new DAP client.
 *
 * This method creates and returns a new DAP client object. The
 * serverVersionMajor and serverVersionMinor parameters should give a 
 * version number which is equal to or lower than the one reported in the 
 * format field of the remotingInfo element in the DAP server's entry of the
 * applications list.
 *
 * Before issuing any attestation requests, the application must license and
 * configure the DAP client. See \ref subsection_usage_licensing and
 * \ref subsection_usage_configuring for further information.
 *
 * After licensing and configuring the DAP client, the application should call
 * VNCDAPClientAttestationRequest() to perform an attestation request. The
 * application may reuse the same DAP client object for as many attestation
 * requests as it wishes.
 *
 * The application may create as many DAP client objects as it wishes (for
 * example, to interact with multiple MirrorLink devices simultaneously).
 * Usually, however, only one DAP client is needed.
 *
 * When a DAP client is no longer needed, the application should destroy it
 * with VNCDAPClientDestroy().
 *
 * \param ppDAPClient On successful return, *ppDAPClient points to the new DAP
 * client object.
 * \param serverVersionMajor The major component of the version number of the
 *  DAP application Remote Access Protocol Format.
 * \param serverVersionMinor The minor component of the version number of the
 *  DAP application Remote Access Protocol Format.
 * \param pContext A pointer to application-defined data to be associated with
 * the new DAP client. See VNCDAPClientSetContext().
 * \param pCallbacks Pointer to a structure containing callbacks from the SDK
 * into the application. It is recommended that the application provides a
 * VNCDAPClientLogCallback(), so that interoperability issues can be diagnosed.
 * \param callbacksSize Size of the structure pointed to be *pCallbacks.
 *
 * \retval ::VNCDAPErrorNone The DAP client was created successfully.
 * \retval ::VNCDAPErrorServerVersionNotSupported This release of the DAP SDK
 * does not support DAP servers of the specified version.
 *
 * \see VNCDAPClientDestroy(), VNCDAPClientAttestationRequest(),
 * VNCDAPClientSetContext()
 */
typedef VNCDAPError VNCCALL
VNCDAPClientCreate(VNCDAPClient **ppDAPClient,
                   vnc_int32_t serverVersionMajor,
                   vnc_int32_t serverVersionMinor,
                   void *pContext,
                   const VNCDAPClientCallbacks *pCallbacks,
                   size_t callbacksSize);

/**
 * \brief Destroys a DAP client that has been created by VNCDAPClientCreate().
 *
 * \param pDAPClient The DAP client to be destroyed.
 *
 * \retval ::VNCDAPErrorNone The DAP client was destroyed successfully.
 * \retval ::VNCDAPErrorInvalidParameter pDAPClient is NULL.
 *
 * \see VNCDAPClientCreate().
 */
typedef VNCDAPError VNCCALL
VNCDAPClientDestroy(VNCDAPClient *pDAPClient);

/**
 * \brief Associates a context pointer with a DAP client.
 *
 * The content pointer may be used by the application for any purpose that it
 * wishes. The pointer can be retrieved by a call to VNCDAPClientGetContext().
 *
 * The SDK does not interact in any way with the memory to which the pointer
 * points. In particular, the SDK does not free the memory when the DAP client
 * is destroyed.
 * 
 * \param pDAPClient The DAP client.
 * \param pContext The context pointer.
 *
 * \see VNCDAPClientGetContext()
 */
typedef void VNCCALL
VNCDAPClientSetContext(VNCDAPClient *pDAPClient,
                       void *pContext);

/**
 * \brief Returns the context pointer associated with the DAP client.
 *
 * The value returned is value supplied in the last call to
 * VNCDAPClientSetContext() for this DAP client. If VNCDAPClientSetContext()
 * has not been called for this DAP client, then the value returned is the
 * value passed in the the pContext parameter to VNCDAPClientCreate().
 * 
 * \param pDAPClient The DAP client.
 *
 * \return The context pointer associated with the DAP client.
 *
 * \see VNCDAPClientCreate(), VNCDAPClientSetContext()
 */
typedef void *VNCCALL
VNCDAPClientGetContext(VNCDAPClient *pDAPClient);

/**
 * \brief Adds a license to a DAP client.
 *
 * DAP clients must always be licensed before they can be used. It is
 * recommended that VNCDAPClientAddLicense() is called soon after each call to
 * VNCDAPClientCreate().
 *
 * To obtain a license for use with VNC DAP SDK, contact your RealVNC sales
 * representative.
 * 
 * \param pDAPClient The DAP client.
 * \param licenseText The text of the license, as a NUL-terminated ASCII
 * string.
 * \param serialNumber On successful return, if serialNumber is not NULL, then
 * it will contain the serial number of the added license. The serial number is
 * a big-endian UUID.
 * \param serialNumberSize If serialNumber is not NULL, then serialNumberSize
 * should be the size of the buffer pointed to by serialNumber, which should be
 * at least 16 bytes. If serialNumber is NULL, then serialNumberSize is
 * ignored.
 *
 * \retval ::VNCDAPErrorNone The license was added successfully.
 * \retval ::VNCDAPErrorLicenseNotValid licenseText is not a valid VNC license.
 * \retval ::VNCDAPErrorInvalidParameter Either licenseText is NULL, or
 * serialNumber is not NULL and serialNumberSize is less than 16.
 */
typedef VNCDAPError VNCCALL
VNCDAPClientAddLicense(VNCDAPClient *pDAPClient,
                       const char *licenseText,
                       vnc_uint8_t *serialNumber,
                       size_t serialNumberSize);

/**
 * \brief Sets the trust root certificate that will be used to verify received
 * server certificates.
 *
 * The application must call this method successfully before any call to
 * VNCDAPClientAttestationRequest() can be successful.
 *
 * See \ref ml_dap "[ML-DAP]" for further information.
 * 
 * \param pDAPClient The DAP client.
 * \param certificate Buffer containing the X.509 certificate. The supported
 * encodings for the certificate buffer are PEM and DER.
 * \param certificateSize Size of the data in the certificate buffer.
 *
 * \retval ::VNCDAPErrorNone The trust root was set successfully.
 * \retval ::VNCDAPErrorInvalidCertificate The buffer provided does not contain a
 * valid X.509 certificate, or is encoded using some unsupported encoding (e.g.
 * PKCS#12).
 *
 * \see VNCDAPClientAttestationRequest()
 */
typedef VNCDAPError VNCCALL
VNCDAPClientSetTrustRoot(VNCDAPClient *pDAPClient,
                         const vnc_uint8_t *certificate,
                         size_t certificateSize);

/**
 * \brief Configures a timeout for attestation requests.
 *
 * By default, there is no timeout, and VNCDAPClientAttestationRequest() will
 * block the calling thread until it either succeeds or fails.
 * 
 * \param pDAPClient The DAP client.
 * \param timeoutMilliseconds The new timeout in milliseconds. If
 * timeoutMilliseconds is less than 0, then any previously-configured timeout
 * is removed.
 *
 * \retval ::VNCDAPErrorNone The timeout was set successfully.
 */
typedef VNCDAPError VNCCALL
VNCDAPClientSetTimeout(VNCDAPClient *pDAPClient,
                       vnc_int32_t timeoutMilliseconds);

/**
 * \brief Sets the URL of the DAP server that will be used for attestation
 * requests.
 *
 * The application must call this method successfully before any call to
 * VNCDAPClientAttestationRequest() can be successful.
 *
 * See \ref ml_app "[ML-APP]" for details of how to obtain the DAP server URL.
 * If you are using VNC Discovery SDK, then you can obtain the URL by first
 * launching the application and then fetching its VNCDiscoverySDKMLAppUri
 * value.
 * 
 * \param pDAPClient The DAP client.
 * \param dapServerURL The DAP server URL.
 *
 * \retval ::VNCDAPErrorNone The server URL was set successfully.
 * \retval ::VNCDAPErrorInvalidParameter The provided URL is not a valid DAP
 * server URL.
 *
 * \see VNCDAPClientAttestationRequest()
 */
typedef VNCDAPError VNCCALL
VNCDAPClientSetServerURL(VNCDAPClient *pDAPClient,
                         const char *dapServerURL);

/**
 * \brief Disables checking of the validity period of X.509 certificates.
 *
 * This will disable checking of the validity period of certificates in the
 * certificate chain used to attest devices. If required, this should be
 * called before issuing an attestation request.
 *
 * This may be necessary as a workaround for devices without a reliable
 * real-time clock, but should not be used otherwise as it will reduce the
 * security offered by X.509 certificates.
 *
 * This should only be used if the MirrorLink version is 1.2 or below and the
 * client does not have reliable access to the current date.
 *
 * MirrorLink 1.3 (or above) clients shall use
 * VNCDAPClientSetCertificateValidityPeriod() instead.
 *
 * \param pDAPClient The DAP client.
 *
 * \retval ::VNCDAPErrorNone Validity period checking has been disabled.
 * \retval ::VNCDAPErrorFailed Validity period checking cannot be disabled.
 * This is probably due to the fact that a certificate validity date has been
 * previously set by a call to VNCDAPClientSetCertificateValidityPeriod().
 *
 * \see VNCDAPClientSetCertificateValidityPeriod()
 */
typedef VNCDAPError VNCCALL
VNCDAPClientDisableCertificateValidityPeriodCheck(VNCDAPClient *pDAPClient);

/**
 * \brief Sets the earliest allowed expiry date for X.509 certificates.
 *
 * This function allows users to set a timestamp to act as a lower bound on when
 * certificates are allowed to have expired. This is necessary if the system
 * time for the client cannot be accurately set.
 *
 * For example, by using a timestamp set to '1578182400' (which corresponds to
 * 00:00, Jan 1 2020 UTC), the SDK will consider valid only DAP servers'
 * certificates having a "Not After" validity period field after 00:00, Jan 1
 * 2020 UTC.
 *
 * According to the MirrorLink 1.3 specification this timestamp should be no
 * earlier than 6 months prior to seeking MirrorLink certification.
 *
 * This should only be used if the MirrorLink version is 1.3 or above and the
 * client does not have reliable access to the current date.
 *
 * This API replaces VNCDAPClientDisableCertificateValidityPeriodCheck() for
 * MirrorLink 1.3 (and above) clients.
 *
 * \param pDAPClient The DAP client.
 * \param notAfter The timestamp before which X.509 certificates must not have
 * expired. This shall be compared against the timestamp in the certificate's
 * "Not After" validity period field. If the certificate has an earlier
 * timestamp, then it shall be considered invalid. This must hold the number of
 * seconds (not counting leap seconds) since 00:00, Jan 1 1970 UTC,
 * corresponding to POSIX time.
 *
 * \retval ::VNCDAPErrorNone The certificate validity period has been set
 * successfully.
 * \retval ::VNCDAPErrorInvalidParameter pDAPClient is NULL or the notAfter
 * value is not valid.
 *
 * \see VNCDAPClientDisableCertificateValidityPeriodCheck()
 */
typedef VNCDAPError VNCCALL
VNCDAPClientSetCertificateValidityPeriod(VNCDAPClient *pDAPClient,
                                         vnc_uint64_t notAfter);

/**
 * \brief Issues a single &lt;attestationRequest&gt; to the DAP server, then
 * parses, verifies and returns the &lt;attestationResponse&gt;.
 *
 * This version of the function performs a blocking request, filling in the
 * response and then returning. For the non-blocking version see
 * VNCDAPClientAttestationRequestEx().
 *
 * The application must license and configure the DAP client before this method
 * can be called. See \ref subsection_usage_licensing and
 * \ref subsection_usage_configuring for further information.
 *
 * If the DAP client is not already connected to the DAP server, then this
 * method will automatically establish a connection. The DAP client will then
 * remain connected to the DAP server until the application calls
 * VNCDAPClientDisconnect(), or until the server closes the connection, or
 * there is some other network error.
 *
 * The componentId parameter should normally be one of the \ref VNCDAPComponent
 * constants. If componentId is VNCDAPComponentAll (the "*" wildcard), then the
 * DAP server should attest every MirrorLink component the MirrorLink server
 * supports. Otherwise, the DAP server should attest only the requested
 * component.
 *
 * VNC DAP SDK perform full verification of the received
 * &lt;attestationResponse&gt;. The verification succeeds only if all of the
 * following are true:
 *
 *  - the XML is well-formed
 *  - the XML is a valid &lt;attestationResponse&gt; according to the schema
 *  - the device certificate is a well-formed Base-64-and-DER-encoded X.509
 *    certificate
 *  - any manufacturer certificates are well-formed Base-64-and-DER-encoded
 *    X.509 certificates
 *  - the device certificate and manufacturer certificates form a chain of
 *    trust ending in the trust root
 *  - the device certificate and any manufacturer certificates within the
 *    chain of trust are within their periods of validity
 *  - the device certificate has the X.509 version 3 extension 'extended key
 *    usage' equal to tcg-kp-AIKCertificate
 *  - any manufacturer certificates in the chain of trust have the X.509
 *    version 3 extension 'basic contraints' equal to CA:TRUE
 *  - the &lt;quoteInfo&gt; in each &lt;attestation&gt; is equal to a
 *    TPM_QUOTE_INFO calculated by the SDK
 *  - the &lt;quoteSignature&gt; in each &lt;attestation&gt; is a SHA1-with-RSA
 *    signature of the &lt;quoteInfo&gt; by the private key matching the device
 *    certificate
 *  - (if the componentId parameter was not VNCDAPComponentAll) the
 *    &lt;attestationResponse&gt; contains an &lt;attestation&gt; for the
 *    requested component
 *
 * Failures in validation are reported through the VNCDAPClientLogCallback() as
 * well as via the error code.
 *
 * See \ref ml_dap "[ML-DAP]" for a full description of the DAP protocol.
 *
 * \param pDAPClient The DAP client.
 * \param componentId The component identifier of the MirrorLink component that
 * the server should attest.
 * \param ppAttestationResponse On successful return, points to a
 * ::VNCDAPAttestationResponse structure containing the parsed response. The
 * application must free this structure by calling
 * VNCDAPAttestationResponseDestroy() when it is no longer required. (It is
 * legal for the application to destroy the DAP client before destroying the
 * attestation response.)
 * \param pAttestationResponseSize On successful return,
 * *pAttestationResponseSize is the size of the structure pointed to by
 * *ppAttestationResponse.
 *
 * \retval ::VNCDAPErrorNone Attestation was completely successful.
 * \retval ::VNCDAPErrorFeatureNotLicensed No license has been added to the DAP
 * client (or a license was added to the DAP client but has expired.)
 * \retval ::VNCDAPErrorNoServerURL No DAP server URL has been configured.
 * \retval ::VNCDAPErrorNoTrustRoot No trust root certificate has been
 * configured.
 * \retval ::VNCDAPErrorInvalidParameter componentId, pAttestationResponse or
 * pAttestationResponseSize is NULL.
 * \retval ::VNCDAPErrorNetworkError A connection to the DAP server could not be
 * established, or the DAP server disconnected while the SDK was attempting to
 * read the attestation response.
 * \retval ::VNCDAPResultComponentNotImplemented The DAP server does not
 * implement the MirrorLink component identified by componentId.
 * \retval ::VNCDAPResultUnknownTrustRoot The DAP server does not recognize the
 * configured trust root certificate.
 * \retval ::VNCDAPResultVersionUnsupported The DAP server does not support the
 * version of the DAP protocol that was specified in the call to
 * VNCDAPClientCreate().
 * \retval ::VNCDAPResultUnknownComponent The DAP server does not recognize the
 * componentId.
 * \retval ::VNCDAPResultAttestationFailed The DAP server could not complete the
 * attestation.
 * \retval ::VNCDAPErrorInvalidAttestationResponse The XML received from the DAP
 * server, or one of the CDATA values in the XML, could not be parsed or is not
 * valid.
 * \retval ::VNCDAPErrorInvalidCertificate The DAP server returned a device
 * certificate or a manufacturer certificate that is not a valid X.509
 * certificate.
 * \retval ::VNCDAPErrorServerNotTrusted The SDK's verification of the
 * attestation response failed either because no trusted certificate chain
 * could be created or because the server's TPM_QUOTE_INFO or
 * TPM_QUOTE_SIGNATURE was incorrect.
 *
 * \see VNCDAPClientDisconnect(), VNCDAPAttestationResponseDestroy()
 */
typedef VNCDAPError VNCCALL
VNCDAPClientAttestationRequest(
        VNCDAPClient *pDAPClient,
        const char *componentId,
        VNCDAPAttestationResponse **ppAttestationResponse,
        size_t *pAttestationResponseSize);


/** \brief As VNCDAPClientAttestationRequest() but non-blocking.
 *
 * The operation of this function is identical to
 * VNCDAPClientAttestationRequest(), but occurs in the background and the
 * response is returned via VNCDAPClientAttestationResponseCallback().
 *
 * \retval ::VNCDAPErrorNone Attestion request was successfully submitted.
 * \retval ::VNCDAPErrorInvalidParameter componentId is NULL.
 * \retval ::VNCDAPErrorFailed Internal error submitting request.
 */
typedef VNCDAPError VNCCALL
VNCDAPClientAttestationRequestEx(
        VNCDAPClient *pDAPClient,
        const char *componentId);

/**
 * \brief Destroys a VNCDAPAttestationResponse created by
 * VNCDAPClientAttestationRequest().
 * 
 * \param pAttestationResponse The attestation response to be destroyed.
 *
 * \retval ::VNCDAPErrorNone The attestation response was successfully destroyed.
 * \retval ::VNCDAPErrorInvalidParameter pAttestationResponse is NULL.
 *
 * \see VNCDAPClientAttestationRequest()
 */
typedef VNCDAPError VNCCALL
VNCDAPAttestationResponseDestroy(
        VNCDAPAttestationResponse *pAttestationResponse);

/**
 * \brief Disconnects the DAP client from the DAP server.
 *
 * The DAP client will automatically disconnect from the server DAP server
 * before is destroyed by VNCDAPClientDestroy(). However, if you do not plan to
 * destroy a DAP client immediately, it is recommended that you disconnect
 * explicitly by calling VNCDAPClientDisconnect().
 *
 * The DAP client will automatically reconnect to the DAP server if
 * VNCDAPClientAttestationRequest() is called again.
 * 
 * \param pDAPClient The DAP client.
 *
 * \retval ::VNCDAPErrorNone The DAP client has disconnected from the DAP server
 * (or was not connected to begin with).
 */
typedef VNCDAPError VNCCALL
VNCDAPClientDisconnect(VNCDAPClient *pDAPClient);

/**
 * \brief Contains the SDK's version and the addresses of the SDK's methods.
 * 
 * \see VNCDAPSDKInitialize()
 */
typedef struct
{
    /** Major component of the SDK's version number. */
    vnc_int32_t versionMajor;
    /** Minor component of the SDK's version number. */
    vnc_int32_t versionMinor;
    /** Patch component of the SDK's version number. */
    vnc_int32_t versionPatch;
    /** Build number component of the SDK's version number. */
    vnc_int32_t versionBuild;

    /** Uninitializes the SDK. */
    VNCDAPSDKUninitialize *vncDAPSDKUninitialize;
    /** Returns a symbolic name corresponding to a DAP SDK error code. */
    VNCDAPGetErrorName *vncDAPGetErrorName;
    /** Creates a new DAP client. */
    VNCDAPClientCreate *vncDAPClientCreate;
    /** Destroys a DAP client that has been created by VNCDAPClientCreate(). */
    VNCDAPClientDestroy *vncDAPClientDestroy;
    /** Associates a context pointer with a DAP client. */
    VNCDAPClientSetContext *vncDAPClientSetContext;
    /** Returns the context pointer associated with the DAP client. */
    VNCDAPClientGetContext *vncDAPClientGetContext;
    /** Adds a license to a DAP client. */
    VNCDAPClientAddLicense *vncDAPClientAddLicense;
    /**
     * Sets the trust root certificate that will be used to verify received
     * server certificates.
     */
    VNCDAPClientSetTrustRoot *vncDAPClientSetTrustRoot;
    /**
     * Sets the URL of the DAP server that will be used for attestation
     * requests.
     */
    VNCDAPClientSetServerURL *vncDAPClientSetServerURL;
    /** Disconnects the DAP client from the DAP server. */
    VNCDAPClientDisconnect *vncDAPClientDisconnect;
    /**
     * Issues a single &lt;attestationRequest&gt; to the DAP server, then
     * parses, verifies and returns the &lt;attestationResponse&gt;.
     */
    VNCDAPClientAttestationRequest *vncDAPClientAttestationRequest;
    /**
     * Destroys a VNCDAPAttestationResponse created by
     * VNCDAPClientAttestationRequest().
     */
    VNCDAPAttestationResponseDestroy *vncDAPAttestationResponseDestroy;
    /** Configures a timeout for attestation requests. */
    VNCDAPClientSetTimeout *vncDAPClientSetTimeout;
    /** Non-blocking version of VNCDAPClientAttestationRequest. */
    VNCDAPClientAttestationRequestEx *vncDAPClientAttestationRequestEx;
    /** Disables checking of the validity period of X.509 certificates. */
    VNCDAPClientDisableCertificateValidityPeriodCheck *vncDAPClientDisableCertificateValidityPeriodCheck;
    /** Sets the earliest allowed expiry date for X.509 certificates. */
    VNCDAPClientSetCertificateValidityPeriod *vncDAPClientSetCertificateValidityPeriod;
} VNCDAPSDK;

/**
 * \brief The type of VNCDAPSDKInitialize().
 *
 * If you are dynamically loading the SDK at runtime, you should use this
 * typedef to declare your function pointer.
 *
 * \see VNCDAPSDKInitialize()
 */
typedef VNCDAPError VNCCALL
VNCDAPSDKInitializeType(VNCDAPSDK *pSDK, size_t sdkSize);

/**
 * \brief The entry point to the SDK.
 *
 * This function is the sole entry point exported by the SDK.  Before you can
 * call any other of the SDK's functions, you must call this function to obtain
 * pointers to their implementations.
 *
 * Before unloading the SDK, you should call VNCDAPSDKUninitialize().
 *
 * \param pSDK Pointer to the VNCDAPSDK structure to be initialized.
 * \param sdkSize Size of the VNCDAPSDK structure to be initialized.
 *
 * \retval ::VNCDAPErrorNone The SDK was initialized successfully.
 * \retval ::VNCDAPErrorInvalidParameter sdkSize is not valid.
 *
 * \see ::VNCDAPSDK, VNCDAPSDKUninitialize(), ::VNCDAPSDKInitializeType,
 * VNCVerifyIntegerTypes()
 */
VNCDLLIMPORT VNCDAPError VNCCALL
VNCDAPSDKInitialize(VNCDAPSDK *pSDK, size_t sdkSize);

#ifdef __cplusplus
}
#endif

#endif /* !defined(UUID_324cd44a_4435_4288_b4fb_45b83430e2e7) */
